import React, { useMemo } from "react";
import {
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell,
  Tooltip,
  LineChart,
  Line,
  XAxis,
  YAxis,
  CartesianGrid,
} from "recharts";

const COLORS = [
  "#6366F1",
  "#F97316",
  "#60A5FA",
  "#F43F5E",
  "#FB7185",
  "#34D399",
  "#FBBF24",
];

export default function Analytics({ items }) {
  // ====== Category wise spending ======
  const byCategory = useMemo(() => {
    const map = {};
    items.forEach((i) => {
      const c = i.category || "General";
      map[c] = (map[c] || 0) + Math.abs(i.amount);
    });
    return Object.entries(map).map(([name, value]) => ({ name, value }));
  }, [items]);

  // ====== Monthly trend (1st of each month) ======
  const trend = useMemo(() => {
    const map = {};
    items.forEach((i) => {
      const date = new Date(i.date || new Date());
      const year = 2025; // fixed to current year
      const month = date.getMonth(); // 0 = Jan, 9 = Oct
      const monthKey = `${year}-${String(month + 1).padStart(2, "0")}-01`; // 1st of each month
      map[monthKey] = (map[monthKey] || 0) + i.amount;
    });

    // Sort by month key (chronologically)
    const sorted = Object.entries(map)
      .sort((a, b) => new Date(a[0]) - new Date(b[0]))
      .map(([date, amt]) => ({ date, amt }));

    return sorted;
  }, [items]);

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
      {/* ===== Pie Chart ===== */}
      <div className="h-48 bg-transparent p-2">
        <h4 className="text-sm font-semibold mb-2">Spending by Category</h4>
        <ResponsiveContainer width="100%" height="100%">
          <PieChart>
            <Pie
              data={byCategory}
              dataKey="value"
              nameKey="name"
              innerRadius={40}
              outerRadius={70}
              paddingAngle={4}
            >
              {byCategory.map((entry, index) => (
                <Cell key={index} fill={COLORS[index % COLORS.length]} />
              ))}
            </Pie>
            <Tooltip formatter={(v) => `₹${v}`} />
          </PieChart>
        </ResponsiveContainer>
      </div>

      {/* ===== Line Chart: Monthly Trend ===== */}
      <div className="h-48 bg-transparent p-2">
        <h4 className="text-sm font-semibold mb-2">Balance Trend</h4>
        <ResponsiveContainer width="100%" height="100%">
          <LineChart data={trend}>
            <CartesianGrid strokeDasharray="3 3" />
            <XAxis
              dataKey="date"
              tickFormatter={(d) =>
                new Date(d).toLocaleString("default", { month: "short" })
              }
            />
            <YAxis />
            <Tooltip
              labelFormatter={(d) =>
                new Date(d).toLocaleDateString("en-IN", {
                  month: "long",
                  year: "numeric",
                })
              }
              formatter={(v) => [`₹${v}`, "Net Amount"]}
            />
            <Line
              type="monotone"
              dataKey="amt"
              stroke="#7c3aed"
              strokeWidth={2}
            />
          </LineChart>
        </ResponsiveContainer>
      </div>
    </div>
  );
}
