import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";

export default function AddTransactionModal({ onClose, onAdd }) {
  const [form, setForm] = useState({
    title: "",
    amount: "",
    category: "General",
    date: "",
    note: "",
  });

  const categories = [
    "General",
    "Food",
    "Travel",
    "Bills",
    "Shopping",
    "Entertainment",
    "Salary",
    "Health",
  ];

  const handle = (e) => setForm({ ...form, [e.target.name]: e.target.value });
  const today = new Date().toISOString().split("T")[0];

  const submit = (e) => {
    e.preventDefault();
    if (!form.title || !form.amount)
      return alert("Please add title and amount");
    onAdd({ ...form, id: Date.now(), amount: parseFloat(form.amount) });
    onClose();
  };

  return (
    <AnimatePresence>
      <motion.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        exit={{ opacity: 0 }}
        transition={{ duration: 0.25 }}
        className="fixed inset-0 z-50 flex items-center justify-center bg-black/40 backdrop-blur-sm p-4"
      >
        <motion.div
          initial={{ scale: 0.9, opacity: 0 }}
          animate={{ scale: 1, opacity: 1 }}
          exit={{ scale: 0.9, opacity: 0 }}
          transition={{ duration: 0.25 }}
          className="w-full sm:max-w-lg bg-white dark:bg-[#04172a] rounded-2xl p-6 shadow-2xl"
        >
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-semibold">Add Transaction</h3>
            <button
              onClick={onClose}
              className="text-gray-500 hover:text-red-500 transition"
            >
              ✕
            </button>
          </div>

          <form
            onSubmit={submit}
            className="grid grid-cols-1 sm:grid-cols-2 gap-3"
          >
            <input
              name="title"
              value={form.title}
              onChange={handle}
              placeholder="Title"
              className="p-2 rounded border dark:bg-[#021427]"
            />

            <input
              name="amount"
              value={form.amount}
              onChange={handle}
              type="number"
              placeholder="Amount (use negative for expense)"
              className="p-2 rounded border dark:bg-[#021427]"
            />

            <select
              name="category"
              value={form.category}
              onChange={handle}
              className="p-2 rounded border dark:bg-[#021427]"
            >
              {categories.map((c) => (
                <option key={c}>{c}</option>
              ))}
            </select>

            <input
              name="date"
              value={form.date}
              onChange={handle}
              type="date"
              max={today}
              className="p-2 rounded border dark:bg-[#021427]"
            />

            <textarea
              name="note"
              value={form.note}
              onChange={handle}
              placeholder="Note (optional)"
              className="p-2 rounded border dark:bg-[#021427] col-span-1 sm:col-span-2"
            />

            <div className="flex gap-2 justify-end col-span-1 sm:col-span-2 mt-3">
              <button
                type="button"
                onClick={onClose}
                className="px-4 py-2 rounded border hover:bg-gray-100 dark:hover:bg-[#02203d] transition"
              >
                Cancel
              </button>
              <button
                type="submit"
                className="px-4 py-2 rounded bg-primary text-white hover:opacity-90 transition"
              >
                Add Transaction
              </button>
            </div>
          </form>
        </motion.div>
      </motion.div>
    </AnimatePresence>
  );
}
